<?php
// uber_cancel_webhook.php
//
// This endpoint is designed to receive Uber Eats webhooks for
// cancellation events (e.g. orders.cancel).
//
// Behaviour NOW:
//   - logs the raw webhook
//   - extracts the Uber order id
//   - builds the body you would POST to your cancel workflow
//     (or directly to Tookan's get_job_details_by_order_id / cancel_job)
//   - logs that body as a DRY RUN
//
// No calls are made to Tookan yet.

declare(strict_types=1);

require_once __DIR__ . '/uber_config.php';

// --- Read and log raw request ---
$rawBody = file_get_contents('php://input') ?: '';
$headers = function_exists('getallheaders') ? getallheaders() : [];

uberLog('UBER_WEBHOOK_INCOMING_CANCEL', [
    'headers' => $headers,
    'raw' => $rawBody,
]);

if ($rawBody === '') {
    uberJsonResponse(400, ['error' => 'Empty request body']);
}

$payload = json_decode($rawBody, true);
if (!is_array($payload)) {
    uberJsonResponse(400, ['error' => 'Invalid JSON payload']);
}

$eventType = $payload['event_type'] ?? '';
$resourceHref = $payload['resource_href'] ?? null;
$meta = $payload['meta'] ?? [];
$uberId = $meta['resource_id'] ?? null;

if ($eventType !== 'orders.cancel') {
    // Log but don't fail. This endpoint might get other events too,
    // depending on how you configure Uber.
    uberLog('UBER_CANCEL_ENDPOINT_NON_CANCEL', [
        'event_type' => $eventType,
        'uber_id' => $uberId,
        'resource_href' => $resourceHref,
    ]);

    uberJsonResponse(200, [
        'message' => 'Non-cancel event received on cancel endpoint; logged only',
        'event_type' => $eventType,
        'uber_id' => $uberId,
    ]);
}

if (!$uberId) {
    uberJsonResponse(400, ['error' => 'Missing meta.resource_id for cancel']);
}

// We'll keep the order id for Tookan the same as Uber's resource_id,
// matching the create_webhook logic.
$orderIdForTookan = $uberId;

// This is the body you would POST to your cancel-task.php endpoint,
// or directly to Tookan via get_job_details_by_order_id + cancel_job.
$cancelBody = [
    'apiKey' => UBER_SKED_API_KEY,
    'orderId' => $orderIdForTookan,
];

uberLog('UBER_CANCEL_TOOKAN_DRY_RUN', [
    'uber_id' => $uberId,
    'cancel_body' => $cancelBody,
]);

// --- Notify ntfy.sh ---
$ntfyUrl = 'https://ntfy.sh/' . NTFY_TOPIC;
$ntfyMsg = 'skeduber: Uber Order UE:' . $orderIdForTookan . ' was cancelled';
$ntfyOpts = [
    'http' => [
        'method' => 'POST',
        'header' => "Authorization: Bearer " . NTFY_TOKEN . "\r\n",
        'content' => $ntfyMsg
    ]
];
@file_get_contents($ntfyUrl, false, stream_context_create($ntfyOpts));

uberLog('UBER_CANCEL_NOTIFICATION_SENT', [
    'uber_id' => $uberId,
    'message' => $ntfyMsg,
]);

uberJsonResponse(200, [
    'message' => 'Uber cancel received, notification sent',
    'event_type' => $eventType,
    'uber_id' => $uberId,
    'order_id' => $orderIdForTookan,
]);
