<?php
// uber_create_webhook.php
//
// This endpoint is designed to receive Uber Eats webhooks for
// "create / notification" style events (e.g. orders ready to dispatch).
//
// Behaviour NOW:
//   - logs the raw webhook
//   - fetches order details from Uber using resource_href
//   - extracts a clean summary
//   - builds the FULL Tookan create_task payload (as you would send it)
//   - logs that payload instead of sending it (DRY RUN)
//
// Later, when you're happy with the mapping, you can take the exact
// payload from the DRY RUN log and wire it into your real Tookan call
// or pipe it through create-pick-delivery-task.php.

declare(strict_types=1);

require_once __DIR__ . '/uber_helpers.php';

// --- Read and log raw request ---
$rawBody = file_get_contents('php://input') ?: '';
$headers = function_exists('getallheaders') ? getallheaders() : [];

uberLog('UBER_WEBHOOK_INCOMING_CREATE', [
    'headers' => $headers,
    'raw' => $rawBody,
]);

if ($rawBody === '') {
    uberJsonResponse(400, ['error' => 'Empty request body']);
}

$payload = json_decode($rawBody, true);
if (!is_array($payload)) {
    uberJsonResponse(400, ['error' => 'Invalid JSON payload']);
}

$eventType = $payload['event_type'] ?? '';
$resourceHref = $payload['resource_href'] ?? null;
$meta = $payload['meta'] ?? [];
$uberId = $meta['resource_id'] ?? null;

if (!$resourceHref || !$uberId) {
    uberJsonResponse(400, [
        'error' => 'Missing resource_href or meta.resource_id',
    ]);
}

// Ignore events you don't care about here if needed.
// Example: orders.release is ignored like in your old Deluge script.
if ($eventType === 'orders.release') {
    uberLog('UBER_IGNORED_EVENT', [
        'reason' => 'orders.release ignored on create endpoint',
        'event_type' => $eventType,
        'uber_id' => $uberId,
    ]);

    uberJsonResponse(200, [
        'message' => 'Ignored orders.release event',
        'event_type' => $eventType,
        'uber_id' => $uberId,
    ]);
}

// --- Get OAuth token ---
$accessToken = getUberAccessToken();
if (!$accessToken) {
    uberJsonResponse(500, ['error' => 'Could not obtain Uber access token']);
}

// --- Fetch order details from Uber ---
$order = fetchUberOrderDetails($resourceHref, $accessToken);
if (!$order) {
    uberJsonResponse(500, ['error' => 'Failed to fetch order details']);
}

// --- Extract a clean summary for logging ---
$type = $order['type'] ?? null;
$eater = $order['eater'] ?? [];
$delivery = is_array($eater) ? ($eater['delivery'] ?? []) : [];
$location = is_array($delivery) ? ($delivery['location'] ?? []) : [];
$charges = $order['payment']['charges'] ?? [];

$estimatedReady = $order['estimated_ready_for_pickup_at'] ?? null;

$summary = [
    'uber_id' => $uberId,
    'event_type' => $eventType,
    'resource_href' => $resourceHref,
    'order_type' => $type,
    'environment' => UBER_ENVIRONMENT,

    'eater_first_name' => $eater['first_name'] ?? null,
    'eater_last_name' => $eater['last_name'] ?? null,
    'eater_phone' => $eater['phone'] ?? null,

    'delivery_type' => $delivery['type'] ?? null,
    'delivery_notes' => $delivery['notes'] ?? null,
    'business_name' => $delivery['business_name'] ?? null,
    'unit_number' => $delivery['unit_number'] ?? null,

    'location_type' => $location['type'] ?? null,
    'lat' => $location['latitude'] ?? null,
    'lng' => $location['longitude'] ?? null,
    'street_address' => $location['street_address'] ?? null,
    'google_place_id' => $location['google_place_id'] ?? null,

    'estimated_ready_for_pickup_at' => $estimatedReady,

    'sub_total_amount' => $charges['sub_total']['amount'] ?? null,
    'tip_amount' => $charges['tip']['amount'] ?? null,
    'currency' => $charges['sub_total']['currency_code'] ?? null,
];

uberLog('UBER_CREATE_SUMMARY', $summary);

// ----------------------------------------------------------------
// Build the FULL Tookan create_task payload as if we were about to
// call https://api.tookanapp.com/v2/create_task.
//
// IMPORTANT: This is DRY RUN ONLY. We do NOT send it anywhere – we
// just log it so you can review the mapping.
// ----------------------------------------------------------------

// Order ID to use in Tookan. Keeping it simple: use the Uber resource_id directly.
$orderIdForTookan = $uberId;

// Customer name + phone
$customerFirst = trim((string) ($summary['eater_first_name'] ?? ''));
$customerLast = trim((string) ($summary['eater_last_name'] ?? ''));
$customerName = trim($customerFirst . ' ' . $customerLast);
if ($customerName === '') {
    $customerName = 'Uber Customer';
}
$customerPhone = $summary['eater_phone'] ?? '';

// Build human-readable address (street + business name + unit if present)
$addrParts = [];
if (!empty($summary['street_address'])) {
    $addrParts[] = $summary['street_address'];
}
if (!empty($summary['business_name'])) {
    $addrParts[] = $summary['business_name'];
}
if (!empty($summary['unit_number'])) {
    $addrParts[] = 'Unit ' . $summary['unit_number'];
}
$customerAddress = trim(implode(', ', $addrParts));

// Fallback if address is missing
if ($customerAddress === '') {
    $customerAddress = 'Uber delivery address not provided';
}

// Lat/lng for delivery
$deliveryLat = $summary['lat'] ?? null;
$deliveryLng = $summary['lng'] ?? null;

// --- Google Maps Resolution ---
if ($summary['location_type'] === 'GOOGLE_PLACE' && !empty($summary['google_place_id'])) {
    $placeId = $summary['google_place_id'];
    $googleUrl = 'https://maps.googleapis.com/maps/api/place/details/json?place_id=' . $placeId . '&key=' . GOOGLE_MAPS_API_KEY;

    $attempts = 0;
    $googleSuccess = false;
    while ($attempts < 3 && !$googleSuccess) {
        $gResponse = @file_get_contents($googleUrl);
        if ($gResponse) {
            $gData = json_decode($gResponse, true);
            if (isset($gData['result'])) {
                $customerAddress = $gData['result']['formatted_address'] ?? $customerAddress;
                $deliveryLat = $gData['result']['geometry']['location']['lat'] ?? $deliveryLat;
                $deliveryLng = $gData['result']['geometry']['location']['lng'] ?? $deliveryLng;
                uberLog('GOOGLE_MAPS_SUCCESS', ['place_id' => $placeId, 'address' => $customerAddress]);
                $googleSuccess = true;
            }
        }
        if (!$googleSuccess) {
            $attempts++;
            sleep(1);
        }
    }

    if (!$googleSuccess) {
        uberLog('GOOGLE_MAPS_FAILURE', ['place_id' => $placeId]);
        // Notify failure
        $ntfyUrl = 'https://ntfy.sh/' . NTFY_TOPIC;
        $ntfyMsg = 'skeduber: ' . $orderIdForTookan . ' Google Place Retrieval failed';
        $ntfyOpts = [
            'http' => [
                'method' => 'POST',
                'header' => "Authorization: Bearer " . NTFY_TOKEN . "\r\n",
                'content' => $ntfyMsg
            ]
        ];
        @file_get_contents($ntfyUrl, false, stream_context_create($ntfyOpts));
    }
}


// Derive pickup + delivery timestamps
try {
    if (!empty($summary['estimated_ready_for_pickup_at'])) {
        $pickupDt = new DateTime($summary['estimated_ready_for_pickup_at']);
    } else {
        $pickupDt = new DateTime('now');
    }
} catch (Exception $e) {
    $pickupDt = new DateTime('now');
}

// Old logic: delivery time = pickup + 140 minutes
$deliveryDt = clone $pickupDt;
$deliveryDt->modify('+140 minutes');

$jobPickupDatetime = $pickupDt->format('Y-m-d H:i:s');
$jobDeliveryDatetime = $deliveryDt->format('Y-m-d H:i:s');

// Description = delivery type + notes
$deliveryType = $summary['delivery_type'] ?? '';
if ($deliveryType === 'DELIVERY_BY_RESTAURANT') {
    $deliveryType = 'DELIVERY';
}
$deliveryTypeText = $deliveryType !== '' ? $deliveryType : 'DELIVERY';
$notes = trim((string) ($summary['delivery_notes'] ?? ''));
$jobDescription = $deliveryTypeText . ': ' . $notes;

// Build special instructions (what you'll surface in Tookan custom fields)
$specialInstructions = $notes !== '' ? $notes : 'No instructions';

// Monetary fields: keep them raw for now
$subTotal = $summary['sub_total_amount'] ?? 0;
$tip = $summary['tip_amount'] ?? 0;

// Custom field payload matching your create-pick-delivery-task meta fields
$metaFields = [
    ['label' => 'Special_Instructions', 'data' => (string) $specialInstructions],
    ['label' => 'Tip_Amount', 'data' => (string) $tip],
    ['label' => 'Delivery_Charges', 'data' => '0'],
    ['label' => 'Discount', 'data' => '0'],
    ['label' => 'Subtotal', 'data' => (string) $subTotal],
    ['label' => 'Stores_Display_Address', 'data' => UBER_DEFAULT_PICKUP_ADDRESS],
    ['label' => 'Stores_Name', 'data' => UBER_DEFAULT_PICKUP_NAME],
    ['label' => 'Commission', 'data' => '0'],
];

// Final Tookan create_task payload (DRY RUN)
$tookanPayload = [
    'notify' => 1,
    'api_key' => UBER_TOOKAN_API_KEY,
    'auto_assignment' => 1,
    'team_id' => UBER_TOOKAN_TEAM_ID,
    'timezone' => UBER_TOOKAN_TIMEZONE,
    'ride_type' => 1,
    'has_pickup' => 1,
    'has_delivery' => 1,
    'tracking_link' => 0,

    'order_id' => $orderIdForTookan,
    'customer_username' => $customerName,
    'customer_phone' => $customerPhone,
    'customer_address' => $customerAddress,
    'latitude' => $deliveryLat,
    'longitude' => $deliveryLng,

    'job_description' => $jobDescription,
    'job_delivery_datetime' => $jobDeliveryDatetime,

    'job_pickup_name' => UBER_DEFAULT_PICKUP_NAME,
    'job_pickup_phone' => UBER_DEFAULT_PICKUP_PHONE,
    'job_pickup_address' => UBER_DEFAULT_PICKUP_ADDRESS,
    'job_pickup_datetime' => $jobPickupDatetime,

    'custom_field_template' => UBER_TOOKAN_TEMPLATE,
    'ignore_validate_otp' => 1,
    'pickup_custom_field_template' => UBER_TOOKAN_TEMPLATE,
    'pickup_meta_data' => $metaFields,
    'meta_data' => $metaFields,
];

uberLog('UBER_CREATE_TOOKAN_PAYLOAD', [
    'uber_id' => $uberId,
    'order_id' => $orderIdForTookan,
    'tookan_payload' => $tookanPayload,
]);

// --- Send to Tookan ---
$tookanUrl = 'https://api.tookanapp.com/v2/create_task';
$tookanJson = json_encode($tookanPayload);
$tookanOpts = [
    'http' => [
        'method' => 'POST',
        'header' => "Content-Type: application/json\r\n",
        'content' => $tookanJson
    ]
];

$attempts = 0;
$tookanSuccess = false;
while ($attempts < 3 && !$tookanSuccess) {
    $tResponse = @file_get_contents($tookanUrl, false, stream_context_create($tookanOpts));
    if ($tResponse) {
        $tData = json_decode($tResponse, true);
        if (isset($tData['status']) && $tData['status'] == 200) {
            uberLog('TOOKAN_CREATE_SUCCESS', ['response' => $tData]);
            $tookanSuccess = true;
        } else {
            uberLog('TOOKAN_CREATE_FAIL_RESPONSE', ['response' => $tData]);
        }
    }

    if (!$tookanSuccess) {
        $attempts++;
        // Notify retry
        if ($attempts == 2) {
            $ntfyUrl = 'https://ntfy.sh/' . NTFY_TOPIC;
            $ntfyMsg = 'skeduber: ' . $orderIdForTookan . ' Uber Try ' . $attempts;
            $ntfyOptsRetry = [
                'http' => [
                    'method' => 'POST',
                    'header' => "Authorization: Bearer " . NTFY_TOKEN . "\r\n",
                    'content' => $ntfyMsg
                ]
            ];
            @file_get_contents($ntfyUrl, false, stream_context_create($ntfyOptsRetry));
        }
        sleep(1);
    }
}

if (!$tookanSuccess) {
    uberLog('TOOKAN_CREATE_FAILURE_FINAL', ['order_id' => $orderIdForTookan]);
}

uberJsonResponse(200, [
    'message' => 'Uber order processed',
    'event_type' => $eventType,
    'uber_id' => $uberId,
    'order_id' => $orderIdForTookan,
    'tookan_sent' => $tookanSuccess
]);
