<?php
// uber_helpers.php
// Shared helper functions for Uber integration (OAuth + order details).

declare(strict_types=1);

require_once __DIR__ . '/uber_config.php';

/**
 * Fetch an OAuth access token from Uber using the client_credentials grant.
 *
 * Scope is locked to 'eats.store.orders.read' for order API access.
 */
function getUberAccessToken(string $scope = 'eats.store.orders.read'): ?string
{
    // For client_credentials, we use the endpoint that worked in diagnostics.
    // Sandbox: https://sandbox-login.uber.com/oauth/v2/token
    // Production: https://login.uber.com/oauth/v2/token
    $tokenUrl = (UBER_ENVIRONMENT === 'sandbox')
        ? 'https://sandbox-login.uber.com/oauth/v2/token'
        : 'https://login.uber.com/oauth/v2/token';

    $postFields = http_build_query([
        'grant_type' => 'client_credentials',
        'client_id' => UBER_CLIENT_ID,
        'client_secret' => UBER_CLIENT_SECRET,
        'scope' => $scope,
    ]);

    $ch = curl_init($tokenUrl);
    curl_setopt_array($ch, [
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => $postFields,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 15,
    ]);

    $raw = curl_exec($ch);
    $err = curl_error($ch);
    $status = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    uberLog('UBER_TOKEN_RESPONSE', [
        'status' => $status,
        'error' => $err,
        'raw' => $raw,
    ]);

    if ($err || $status < 200 || $status >= 300) {
        // If running in CLI or Test Page, print the error for visibility
        if (php_sapi_name() === 'cli' || isset($_GET['type'])) {
            echo "Auth Error [$status]: " . ($err ?: $raw) . "\n";
        }
        return null;
    }

    $decoded = json_decode((string) $raw, true);
    if (!is_array($decoded)) {
        return null;
    }

    return $decoded['access_token'] ?? null;
}

/**
 * Fetch Order Details using the resource_href provided in the webhook.
 */
function fetchUberOrderDetails(string $resourceHref, string $accessToken): ?array
{
    // Fix for Sandbox: The webhook might send "sandbox-api.uber.com" or "api.uber.com",
    // but for Eats Sandbox we must use "test-api.uber.com".
    if (UBER_ENVIRONMENT === 'sandbox') {
        $resourceHref = str_replace(
            ['https://api.uber.com', 'https://sandbox-api.uber.com'],
            'https://test-api.uber.com',
            $resourceHref
        );
    }

    // MOCK FOR TESTING: If the URL is for our specific test order, return fake data.
    // This allows Option 2 (Simulate Webhook) to test the entire flow (Maps -> Tookan)
    // even if we can't create a real order on Uber.
    if (strpos($resourceHref, 'TEST_ORDER_123') !== false) {
        uberLog('UBER_ORDER_DETAILS_MOCK', ['msg' => 'Returning mock data for TEST_ORDER_123']);
        return [
            'id' => 'TEST_ORDER_123',
            'display_id' => 'MOCK-1234',
            'store' => ['id' => 'TEST_STORE_UUID', 'name' => 'Mock Store'],
            'eater' => ['first_name' => 'Test', 'phone' => '+15550000000'],
            'current_state' => 'CREATED',
            'cart' => [
                'items' => [
                    ['title' => 'Test Burger', 'quantity' => 1, 'price' => ['total_price' => ['amount' => 1000]]]
                ]
            ],
            'payment' => ['total' => ['amount' => 1000, 'currency_code' => 'USD']],
            'placed_at' => date('c'),
            'type' => 'DELIVERY_BY_UBER', // or DELIVERY_BY_RESTAURANT
            'location' => [
                // This will trigger the Google Maps logic!
                'type' => 'GOOGLE_PLACE',
                'google_place_id' => 'ChIJ2eUgeAK6j4ARbn5u_wAGqWA', // Standard Google Place ID (Googleplex)
                'address' => ['street_address' => '1600 Amphitheatre Pkwy']
            ]
        ];
    }

    $ch = curl_init($resourceHref);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 20,
        CURLOPT_HTTPHEADER => [
            'Authorization: Bearer ' . $accessToken,
            'Accept: application/json',
        ],
    ]);

    $raw = curl_exec($ch);
    $err = curl_error($ch);
    $status = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    uberLog('UBER_ORDER_DETAILS_RESPONSE', [
        'status' => $status,
        'error' => $err,
        'raw' => $raw,
        'url' => $resourceHref,
    ]);

    if ($err || $status < 200 || $status >= 300) {
        return null;
    }

    $decoded = json_decode((string) $raw, true);
    return is_array($decoded) ? $decoded : null;
}
